﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.ComplexEventProcessing.Adapters;
using Microsoft.ComplexEventProcessing;

using System.IO;
using Newtonsoft.Json.Linq;

using SocialAnalyzer.CommonObjects.Adapter;

namespace SocialAnalyzer
{
    public class TwitterInputAdapter: TypedPointInputAdapter<Tweet>
    {

        
        private StreamReader streamReader;       
        private TwitterConfig config {get; set;}
        private EventShape shape { get; set; }
        private Boolean streamOpen = false;
        
        /// <summary>
        /// Constructor for the input adapter.
        /// </summary>
        /// <param name="configInfo">Configuration details for the input adapter.</param>
        /// <param name="eventShape">Type of the event, e.g., point, edge</param>
        public TwitterInputAdapter(TwitterConfig configInfo,
            EventShape eventShape)
        {
            this.config = configInfo;
            this.shape = eventShape;

        }
        
        /// <summary>
        /// Implements the behaviour when the adapter resumes.
        /// </summary>
        public override void Resume()
        {
            if (streamOpen == false)
            {
                OpenTwitterStream();
            }
            ReadTwitterEvents();
        }

        /// <summary>
        /// Implements the behaviour when the adapter starts.
        /// </summary>
        public override void Start()
        {
            OpenTwitterStream();
            
            ReadTwitterEvents();
        }

        

        private void OpenTwitterStream()
        {
           
            TwitterOAuth auth = new TwitterOAuth();
            streamReader = auth.GetTwitterStream();

           if (streamReader != null)
            {
                streamOpen = true;
            }
        }


        private void ReadTwitterEvents()
        {
            while (!streamReader.EndOfStream)
            {

                var jObject = default(JObject);
                var line = streamReader.ReadLine();
                //System.Console.WriteLine("StreamReader read: " + line);
                try
                {
                    jObject = JObject.Parse(line);
                }
                catch (Exception ex)
                {
                    System.Console.WriteLine("Error while parsing JSON:" + ex.Message);

                }
                var tweet = new Tweet();
                if (jObject.SelectToken("delete") == null)
                {
                    tweet.ID = jObject["id"].Value<Int64>();
                    tweet.Text = jObject["text"].Value<string>();
                    tweet.Source = jObject["source"].Value<string>();
                    tweet.UserDescription = jObject["user"]["description"].Value<string>();
                    tweet.UserFriendsCount = jObject["user"]["friends_count"].Value<Int32>();
                    tweet.UserID = jObject["user"]["id"].Value<Int64>();
                    tweet.UserLang = jObject["user"]["lang"].Value<string>();
                    tweet.UserName = jObject["user"]["name"].Value<string>();
                    tweet.UserTimeZone = jObject["user"]["time_zone"].Value<string>();
                    tweet.UserURL = jObject["user"]["url"].Value<string>();
                    tweet.CreatedAt = jObject["created_at"].Value<string>();
                    tweet.UserLocation = jObject["user"]["location"].Value<string>();
                }
                else
                {
                    tweet.DeleteStatusID = jObject["delete"]["status"]["id"].Value<Int64>();
                    tweet.DeleteUserID = jObject["delete"]["status"]["user_id"].Value<Int64>();
                }

                CreateAndSendTweetEvent(tweet);
            }
        }

        private void CreateAndSendTweetEvent(Tweet tweet)
        {
            // Produce INSERT event
            var currEvent = CreateInsertEvent();
            currEvent.StartTime = DateTimeOffset.Now;
            currEvent.Payload = tweet;
            var result= Enqueue(ref currEvent);
            if (result == EnqueueOperationResult.Full)
            {
                Ready();
                return;
            }
            // Also send a CTI event
            EnqueueCtiEvent(DateTimeOffset.Now);
            
        }

        public override void Stop()
        {
            base.Stop();
            streamReader.Close();
            streamReader.Dispose();            
            streamOpen = false;
        }
        
    }
}
