﻿using System;
using System.IO;
using System.Net;
using System.Security.Cryptography;
using System.Text;



namespace SocialAnalyzer.CommonObjects.Adapter
{
    // Implemented along an example from https://dev.twitter.com/discussions/12758 
    public class TwitterOAuth
    {

        public string authVerifier { get; set; }

        public StreamReader streamReader { get; set; }

        public StreamReader GetTwitterStream()
        {
            
            
            // oauth properties
            var oauth_version = "1.0";
            var oauth_signature_method = "HMAC-SHA1";
            var oauth_nonce = GetNonce();
            var oauth_timestamp = GetTimestamp();

            var resource_url = Twitter.Default.URL;

            var baseString = GetBaseString(oauth_version, oauth_signature_method, oauth_nonce, 
                oauth_timestamp, resource_url);

            var compositeKey = string.Concat(Uri.EscapeDataString(Twitter.Default.ConsumerSecret),
                                "&", Uri.EscapeDataString(Twitter.Default.AccessTokenSecret));


            string oauth_signature = CreateSignature(baseString, compositeKey);



            var authHeader = CreateRequestHeader(oauth_version, oauth_signature_method, oauth_nonce, 
                oauth_timestamp, oauth_signature);

            // make the request

            ServicePointManager.Expect100Continue = false;
            
            var postBody = "track=" + Uri.EscapeDataString(Twitter.Default.TrackParameterValue);

            resource_url += "?" + postBody;

            StreamReader streamReader = MakeRequestToAPI(resource_url, authHeader);

            return streamReader;
        }

        private static StreamReader MakeRequestToAPI(string resource_url, string authHeader)
        {
            HttpWebRequest request = (HttpWebRequest)WebRequest.Create(resource_url);

            request.Headers.Add("Authorization", authHeader);

            request.Method = "POST";

            request.ContentType = "application/x-www-form-urlencoded";

            WebResponse response = request.GetResponse();

            StreamReader streamReader = new StreamReader(response.GetResponseStream());
            return streamReader;
        }

        private static string CreateRequestHeader(string oauth_version, string oauth_signature_method, string oauth_nonce, string oauth_timestamp, string oauth_signature)
        {
            // create the request header

            var headerFormat = "OAuth oauth_nonce=\"{0}\", oauth_signature_method=\"{1}\", " +
                                "oauth_timestamp=\"{2}\", oauth_consumer_key=\"{3}\", " +
                                "oauth_token=\"{4}\", oauth_signature=\"{5}\", " +
                                "oauth_version=\"{6}\"";

            var authHeader = string.Format(headerFormat,
                                Uri.EscapeDataString(oauth_nonce),
                                Uri.EscapeDataString(oauth_signature_method),
                                Uri.EscapeDataString(oauth_timestamp),
                                Uri.EscapeDataString(Twitter.Default.ConsumerKey),
                                Uri.EscapeDataString(Twitter.Default.AccessToken),
                                Uri.EscapeDataString(oauth_signature),
                                Uri.EscapeDataString(oauth_version)
                );
            return authHeader;
        }

        private static string CreateSignature(string baseString, string compositeKey)
        {
            string oauth_signature;

            using (HMACSHA1 hasher = new HMACSHA1(ASCIIEncoding.ASCII.GetBytes(compositeKey)))
            {
                oauth_signature = Convert.ToBase64String(
                hasher.ComputeHash(ASCIIEncoding.ASCII.GetBytes(baseString)));
            }
            return oauth_signature;
        }

        private static string GetBaseString(string oauth_version, string oauth_signature_method, string oauth_nonce, string oauth_timestamp, string resource_url)
        {
            var baseFormat = "oauth_consumer_key={0}&oauth_nonce={1}&oauth_signature_method={2}" +
             "&oauth_timestamp={3}&oauth_token={4}&oauth_version={5}&track={6}";

            var baseString = string.Format(baseFormat, Twitter.Default.ConsumerKey,
                                            oauth_nonce, oauth_signature_method,
                                            oauth_timestamp, Twitter.Default.AccessToken,
                                            oauth_version, Uri.EscapeDataString(Twitter.Default.TrackParameterValue)

             );

            baseString = string.Concat("POST&", Uri.EscapeDataString(resource_url), "&",
                Uri.EscapeDataString(baseString));
            return baseString;
        }

        private static string GetTimestamp()
        {
            var timeSpan = DateTime.UtcNow - new DateTime(1970, 1, 1, 0, 0, 0, 0, DateTimeKind.Utc);

            var oauth_timestamp = Convert.ToInt64(timeSpan.TotalSeconds).ToString();
            return oauth_timestamp;
        }

        private static string GetNonce()
        {
            var oauth_nonce = Convert.ToBase64String(

                new ASCIIEncoding().GetBytes(DateTime.Now.Ticks.ToString()));
            return oauth_nonce;
        }

    

        

    }
}
